import akshare as ak
import pandas as pd
import numpy as np
import matplotlib.pyplot as plt
from datetime import datetime
import warnings
warnings.filterwarnings('ignore')

# 设置中文字体
plt.rcParams["font.family"] = ["SimHei"]
plt.rcParams['axes.unicode_minus'] = False  # 解决负号显示问题

# 基金列表及其代码和市场
funds = {
    "纳斯达克ETF": ("513300", "sh"),  # 5开头对应sh
    "创业板100ETF": ("159915", "sz"),  # 1开头对应sz
    "德国ETF": ("513030", "sh"),
    "日经ETF": ("513520", "sh"),
    "恒生ETF": ("159920", "sz"),
    "豆粕ETF": ("159985", "sz"),
    "黄金ETF": ("518880", "sh"),
    "石油LOF": ("162719", "sz")
}

def get_fund_data(code, market, start_date, end_date):
    """获取基金历史数据并进行预处理"""
    try:
        symbol = f"{market}{code}"
        df = ak.fund_etf_hist_sina(symbol=symbol)
        
        # 重命名列并选择所需字段
        df = df.rename(columns={
            "日期": "date",
            "开盘价": "open",
            "最高价": "high",
            "最低价": "low",
            "收盘价": "close",
            "成交量": "volume"
        })[["date", "open", "high", "low", "close", "volume"]]
        
        # 转换日期格式并筛选时间范围
        df["date"] = pd.to_datetime(df["date"])
        df = df[(df["date"] >= start_date) & (df["date"] <= end_date)]
        df = df.sort_values("date").reset_index(drop=True)
        
        # 计算所需指标
        df["daily_return"] = df["close"].pct_change() * 100  # 当日涨跌幅(%)
        df["20d_return"] = df["close"].pct_change(20) * 100  # 20日涨幅(%)
        
        # 计算均线
        df["ma5"] = df["close"].rolling(window=5).mean()
        df["ma20"] = df["close"].rolling(window=20).mean()
        df["ma60"] = df["close"].rolling(window=40).mean()
        
        # 计算成交金额及均量
        df["amount"] = df["close"] * df["volume"]  # 近似成交金额
        df["amount_ma5"] = df["amount"].rolling(window=5).mean()  # 5日均量
        df["amount_ma20"] = df["amount"].rolling(window=20).mean()  # 20日均量
        
        return df
    except Exception as e:
        print(f"获取{code}数据出错: {e}")
        return None

def check_buy_conditions(fund_df, date):
    """检查买入条件是否满足"""
    if date not in fund_df["date"].values:
        return False
    
    idx = fund_df[fund_df["date"] == date].index[0]
    
    # 检查是否有足够数据计算指标
    if idx < 60:
        return False
    
    # 检查均线多头排列: ma5 > ma20 > ma60
    ma_condition = (fund_df.loc[idx, "ma5"] > fund_df.loc[idx, "ma20"] and 
                   fund_df.loc[idx, "ma20"] > fund_df.loc[idx, "ma60"])
    
    # 检查5日平均成交金额 > 0.8 * 20日日均成交金额
    vol_condition = fund_df.loc[idx, "amount_ma5"]  < 1 * fund_df.loc[idx, "amount_ma20"]
    
    # 检查当天涨幅 < 1%
    price_condition = fund_df.loc[idx, "daily_return"] < 1
    
    return ma_condition and vol_condition and price_condition

def backtest_strategy(fund_data, start_date, end_date):
    """回测策略"""
    # 获取所有交易日
    all_dates = sorted({date for df in fund_data.values() if df is not None for date in df["date"]})
    all_dates = [d for d in all_dates if start_date <= d <= end_date]
    
    # 初始化回测参数
    portfolio = {
        "cash": 100000,  # 初始资金
        "holdings": None,  # 持有基金
        "entry_price": 0,  # 买入价格
        "entry_date": None,  # 买入日期
        "history": []  # 交易历史
    }
    
    for date in all_dates:
        # 收集当日有数据的基金及其20日涨幅
        daily_returns = {}
        for name, df in fund_data.items():
            if df is None or date not in df["date"].values:
                continue
            idx = df[df["date"] == date].index[0]
            if idx >= 20 and not np.isnan(df.loc[idx, "20d_return"]):
                daily_returns[name] = df.loc[idx, "20d_return"]
        
        # 按20日涨幅排序
        sorted_funds = sorted(daily_returns.items(), key=lambda x: x[1], reverse=True)
        
        # 计算当前资产价值
        current_value = portfolio["cash"]
        if portfolio["holdings"] is not None:
            fund_df = fund_data[portfolio["holdings"]]
            if date in fund_df["date"].values:
                idx = fund_df[fund_df["date"] == date].index[0]
                current_price = fund_df.loc[idx, "close"]
                shares = portfolio["cash"] / portfolio["entry_price"]  # 假设全额买入
                current_value = shares * current_price
        
        # 检查卖出条件: 持有收益超过15%
        if portfolio["holdings"] is not None:
            fund_df = fund_data[portfolio["holdings"]]
            if date in fund_df["date"].values:
                idx = fund_df[fund_df["date"] == date].index[0]
                current_price = fund_df.loc[idx, "close"]
                returns = (current_price - portfolio["entry_price"]) / portfolio["entry_price"] * 100
                
                if returns > 15:
                    # 执行卖出
                    portfolio["cash"] = current_value
                    portfolio["history"].append({
                        "date": date,
                        "action": "卖出",
                        "fund": portfolio["holdings"],
                        "price": current_price,
                        "return": returns,
                        "value": portfolio["cash"]
                    })
                    portfolio["holdings"] = None
                    portfolio["entry_price"] = 0
                    portfolio["entry_date"] = None
        
        # 检查买入条件: 没有持仓且有符合条件的基金
        if portfolio["holdings"] is None and sorted_funds:
            top_fund = sorted_funds[0][0]  # 20日涨幅排名第一的基金
            top_fund_df = fund_data[top_fund]
            
            if check_buy_conditions(top_fund_df, date):
                # 执行买入
                idx = top_fund_df[top_fund_df["date"] == date].index[0]
                buy_price = top_fund_df.loc[idx, "close"]
                portfolio["holdings"] = top_fund
                portfolio["entry_price"] = buy_price
                portfolio["entry_date"] = date
                portfolio["history"].append({
                    "date": date,
                    "action": "买入",
                    "fund": top_fund,
                    "price": buy_price,
                    "return": 0,
                    "value": portfolio["cash"]
                })
        
        # 记录每日资产价值
        portfolio["history"].append({
            "date": date,
            "action": "持有",
            "fund": portfolio["holdings"],
            "price": None,
            "return": None,
            "value": current_value
        })
    
    return portfolio

def calculate_metrics(history):
    """计算策略表现指标"""
    # 提取每日资产价值
    value_history = [h for h in history if h["action"] == "持有"]
    if not value_history:
        return {}
    
    dates = [h["date"] for h in value_history]
    values = [h["value"] for h in value_history]
    initial_value = values[0]
    final_value = values[-1]
    
    # 计算累计收益率
    total_return = (final_value - initial_value) / initial_value * 100
    
    # 计算年化收益率
    days = (dates[-1] - dates[0]).days
    years = days / 365.25
    annualized_return = (pow(final_value / initial_value, 1/years) - 1) * 100 if years > 0 else 0
    
    # 计算最大回撤
    peak = values[0]
    max_drawdown = 0
    for value in values:
        if value > peak:
            peak = value
        drawdown = (peak - value) / peak * 100
        if drawdown > max_drawdown:
            max_drawdown = drawdown
    
    # 计算交易统计
    trades = [h for h in history if h["action"] in ["买入", "卖出"]]
    buy_count = sum(1 for h in trades if h["action"] == "买入")
    sell_count = sum(1 for h in trades if h["action"] == "卖出")
    
    # 计算胜率
    winning_trades = [h for h in trades if h["action"] == "卖出" and h["return"] > 0]
    win_rate = len(winning_trades) / sell_count * 100 if sell_count > 0 else 0
    
    # 计算平均收益
    avg_return = np.mean([h["return"] for h in trades if h["action"] == "卖出"]) if sell_count > 0 else 0
    
    return {
        "初始资金": initial_value,
        "最终资金": round(final_value, 2),
        "累计收益率(%)": round(total_return, 2),
        "年化收益率(%)": round(annualized_return, 2),
        "最大回撤(%)": round(max_drawdown, 2),
        "交易次数": buy_count,
        "胜率(%)": round(win_rate, 2),
        "平均收益率(%)": round(avg_return, 2),
        "回测周期": f"{dates[0].date()} 至 {dates[-1].date()}"
    }

def plot_performance(history):
    """绘制收益率曲线"""
    value_history = [h for h in history if h["action"] == "持有"]
    if not value_history:
        print("没有足够数据绘制收益率曲线")
        return
    
    dates = [h["date"] for h in value_history]
    values = [h["value"] for h in value_history]
    initial_value = values[0]
    
    # 绘制资产价值曲线
    plt.figure(figsize=(12, 6))
    plt.plot(dates, values, label="策略资产价值", linewidth=2)
    plt.axhline(y=initial_value, color='r', linestyle='--', label="初始资金")
    
    # 标记买卖点
    buy_points = [h for h in history if h["action"] == "买入"]
    sell_points = [h for h in history if h["action"] == "卖出"]
    
    for bp in buy_points:
        plt.scatter(bp["date"], bp["value"], color='g', marker='^', label="买入" if bp == buy_points[0] else "")
    
    for sp in sell_points:
        plt.scatter(sp["date"], sp["value"], color='r', marker='v', label="卖出" if sp == sell_points[0] else "")
    
    plt.title("策略收益率曲线")
    plt.xlabel("日期")
    plt.ylabel("资产价值 (元)")
    plt.grid(alpha=0.3)
    plt.legend()
    plt.gcf().autofmt_xdate()  # 自动旋转日期标签
    plt.tight_layout()
    plt.show()

def main():
    # 设置回测时间范围
    start_date = datetime(2020, 1, 1)
    end_date = datetime.now()
    
    # 获取所有基金数据
    print("正在获取基金数据...")
    fund_data = {}
    for name, (code, market) in funds.items():
        print(f"获取 {name}({code}) 数据...")
        df = get_fund_data(code, market, start_date, end_date)
        if df is not None and not df.empty:
            fund_data[name] = df
        else:
            print(f"警告: 无法获取 {name} 的有效数据，将跳过该基金")
    
    if not fund_data:
        print("无法获取任何基金数据，程序终止")
        return
    
    # 执行回测
    print("开始策略回测...")
    portfolio = backtest_strategy(fund_data, start_date, end_date)
    
    # 计算并输出绩效指标
    metrics = calculate_metrics(portfolio["history"])
    print("\n策略表现指标:")
    for key, value in metrics.items():
        print(f"{key}: {value}")
    
    # 绘制收益率曲线
    plot_performance(portfolio["history"])

if __name__ == "__main__":
    main()