import akshare as ak
import pandas as pd
import numpy as np
import matplotlib.pyplot as plt
import matplotlib.dates as mdates

# 设置中文显示
plt.rcParams["font.family"] = ["SimHei", "WenQuanYi Micro Hei", "Heiti TC"]
plt.rcParams["axes.unicode_minus"] = False  # 解决负号显示问题

# 获取黄金ETF数据（513500为华安黄金ETF）
fund_etf_hist_em_df = ak.fund_etf_hist_em(
    symbol="512890",
    period="daily",
    start_date="20200101",  # 该ETF上市时间为2014年，调整起始日期
    end_date="20260201",
    adjust="qfq"
)

# 数据预处理
fund_etf_hist_em_df['日期'] = pd.to_datetime(fund_etf_hist_em_df['日期'])
fund_etf_hist_em_df = fund_etf_hist_em_df.sort_values('日期').reset_index(drop=True)

# ---------------------- 策略参数设置 ----------------------
initial_cash = 100000  # 初始资金
base_buy_ratio = 0.1   # 底仓占总资金比例（10%）
pyramid_ratios = [0.2, 0.3, 0.4]  # 金字塔补仓比例（逐层递增）
fall_thresholds = [-0.05, -0.10, -0.15]  # 触发补仓的跌幅（5%、10%、15%）
sell_threshold = 0.10  # 止盈涨幅（10%）
max_pyramid_level = 3  # 最大补仓层数（金字塔共4层：1底仓+3补仓）

# 初始化账户信息
fund_etf_hist_em_df['持仓数量'] = 0
fund_etf_hist_em_df['可用资金'] = initial_cash
fund_etf_hist_em_df['持仓市值'] = 0.0
fund_etf_hist_em_df['总资产'] = initial_cash
fund_etf_hist_em_df['操作'] = '无'

# 记录持仓状态
average_cost = 0.0      # 平均持仓成本
current_level = 0       # 当前持仓层数（0=空仓，1=底仓，2=1次补仓，以此类推）
buy_points = []         # 记录每次买入的价格和数量（用于计算平均成本）

# ---------------------- 策略执行逻辑 ----------------------
for i in range(1, len(fund_etf_hist_em_df)):
    prev_data = fund_etf_hist_em_df.iloc[i-1]
    current_data = fund_etf_hist_em_df.iloc[i].copy()
    
    # 继承前一天的账户状态
    current_data['持仓数量'] = prev_data['持仓数量']
    current_data['可用资金'] = prev_data['可用资金']
    current_data['持仓市值'] = current_data['持仓数量'] * current_data['收盘']
    current_data['总资产'] = current_data['可用资金'] + current_data['持仓市值']
    
    # 计算相对于平均成本的涨跌幅（有持仓时）
    if current_data['持仓数量'] > 0:
        price_change = (current_data['收盘'] - average_cost) / average_cost
    else:
        price_change = 0.0
    
    # 1. 上涨止盈逻辑（达到目标涨幅全部卖出）
    if current_data['持仓数量'] > 0 and price_change >= sell_threshold:
        sell_value = current_data['持仓数量'] * current_data['收盘']
        current_data['可用资金'] += sell_value
        current_data['持仓数量'] = 0
        current_data['持仓市值'] = 0.0
        current_data['总资产'] = current_data['可用资金']
        current_data['操作'] = f'止盈卖出，涨幅{price_change:.2%}'
        # 重置持仓状态
        average_cost = 0.0
        current_level = 0
        buy_points = []
    
    # 2. 金字塔补仓逻辑（未达最大层数且有可用资金）
    elif current_data['持仓数量'] > 0 and current_level < max_pyramid_level:
        # 检查是否达到下一层补仓跌幅
        target_level = current_level + 1
        if price_change <= fall_thresholds[target_level - 1]:
            # 计算补仓金额（按金字塔比例）
            buy_amount = initial_cash * pyramid_ratios[target_level - 1]
            if current_data['可用资金'] >= buy_amount:
                buy_shares = int(buy_amount / current_data['收盘'])  # 取整数股
                if buy_shares > 0:
                    # 记录买入点，用于计算平均成本
                    buy_points.append({
                        'shares': buy_shares,
                        'price': current_data['收盘']
                    })
                    # 计算新的平均成本（加权平均）
                    total_cost = sum(p['shares'] * p['price'] for p in buy_points)
                    total_shares = sum(p['shares'] for p in buy_points)
                    average_cost = total_cost / total_shares
                    
                    # 更新账户信息
                    current_data['持仓数量'] = total_shares
                    current_data['可用资金'] -= buy_shares * current_data['收盘']
                    current_data['持仓市值'] = total_shares * current_data['收盘']
                    current_data['总资产'] = current_data['可用资金'] + current_data['持仓市值']
                    current_data['操作'] = f'金字塔{target_level}层补仓，跌幅{price_change:.2%}'
                    current_level = target_level  # 升级持仓层数
    
    # 3. 首次建仓（空仓时建立底仓）
    elif current_data['持仓数量'] == 0 and i > 20:  # 跳过前20天数据
        buy_amount = initial_cash * base_buy_ratio  # 底仓金额
        buy_shares = int(buy_amount / current_data['收盘'])
        if buy_shares > 0:
            buy_points.append({
                'shares': buy_shares,
                'price': current_data['收盘']
            })
            average_cost = current_data['收盘']  # 初始成本=买入价
            current_data['持仓数量'] = buy_shares
            current_data['可用资金'] -= buy_shares * current_data['收盘']
            current_data['持仓市值'] = buy_shares * current_data['收盘']
            current_data['总资产'] = current_data['可用资金'] + current_data['持仓市值']
            current_data['操作'] = '建立底仓（金字塔1层）'
            current_level = 1  # 底仓为第1层
    
    # 更新当前行数据
    fund_etf_hist_em_df.iloc[i] = current_data

# ---------------------- 可视化分析 ----------------------
# 1. 总资产与黄金ETF价格对比
fig, ax1 = plt.subplots(figsize=(14, 8))

# 左侧轴：总资产
ax1.plot(fund_etf_hist_em_df['日期'], fund_etf_hist_em_df['总资产'], 
         color='tab:blue', label='策略总资产', linewidth=2)
ax1.axhline(y=initial_cash, color='gray', linestyle='--', label='初始资金')
ax1.set_xlabel('日期')
ax1.set_ylabel('总资产（元）', color='tab:blue')
ax1.tick_params(axis='y', labelcolor='tab:blue')
ax1.grid(alpha=0.3)

# 右侧轴：ETF收盘价
ax2 = ax1.twinx()
ax2.plot(fund_etf_hist_em_df['日期'], fund_etf_hist_em_df['收盘'], 
         color='tab:orange', label='黄金ETF收盘价', alpha=0.7, linestyle='--')
ax2.set_ylabel('收盘价（元）', color='tab:orange')
ax2.tick_params(axis='y', labelcolor='orange')

# 合并图例
lines1, labels1 = ax1.get_legend_handles_labels()
lines2, labels2 = ax2.get_legend_handles_labels()
ax1.legend(lines1 + lines2, labels1 + labels2, loc='upper left')

# 格式化日期轴
ax1.xaxis.set_major_formatter(mdates.DateFormatter('%Y-%m'))
plt.setp(ax1.xaxis.get_majorticklabels(), rotation=45)

plt.title('金字塔补仓策略总资产与黄金ETF价格走势')
plt.tight_layout()
plt.show()

# 2. 持仓市值与可用资金变化
fig, ax = plt.subplots(figsize=(14, 8))

ax.plot(fund_etf_hist_em_df['日期'], fund_etf_hist_em_df['持仓市值'], 
        color='tab:green', label='持仓市值', linewidth=1.5)
ax.plot(fund_etf_hist_em_df['日期'], fund_etf_hist_em_df['可用资金'], 
        color='tab:red', label='可用资金', linewidth=1.5)
ax.plot(fund_etf_hist_em_df['日期'], fund_etf_hist_em_df['总资产'], 
        color='tab:blue', label='总资产', linewidth=2, linestyle='--')

ax.set_xlabel('日期')
ax.set_ylabel('金额（元）')
ax.legend(loc='upper left')
ax.grid(alpha=0.3)

# 格式化日期轴
ax.xaxis.set_major_formatter(mdates.DateFormatter('%Y-%m'))
plt.setp(ax.xaxis.get_majorticklabels(), rotation=45)

plt.title('持仓市值、可用资金与总资产动态变化')
plt.tight_layout()
plt.show()

# 3. 标记金字塔操作点（底仓、补仓、止盈）
fig, ax = plt.subplots(figsize=(14, 8))

# 绘制价格曲线
ax.plot(fund_etf_hist_em_df['日期'], fund_etf_hist_em_df['收盘'], 
        color='tab:gray', label='收盘价', alpha=0.6)

# 标记操作点
actions = fund_etf_hist_em_df[fund_etf_hist_em_df['操作'] != '无']
for _, row in actions.iterrows():
    if '底仓' in row['操作']:
        ax.scatter(row['日期'], row['收盘'], color='green', s=80, marker='^', label='底仓（1层）')
    elif '补仓' in row['操作']:
        # 根据补仓层数调整大小（层数越高点越大）
        level = int(row['操作'].split('层')[0].split('塔')[1])
        ax.scatter(row['日期'], row['收盘'], color='blue', s=50 + level*30, marker='o', 
                  label=f'补仓{level}层' if f'补仓{level}层' not in ax.get_legend_handles_labels()[1] else "")
    elif '止盈' in row['操作']:
        ax.scatter(row['日期'], row['收盘'], color='red', s=80, marker='v', label='止盈卖出')

# 去重图例
handles, labels = ax.get_legend_handles_labels()
by_label = dict(zip(labels, handles))
ax.legend(by_label.values(), by_label.keys(), loc='upper left')

ax.set_xlabel('日期')
ax.set_ylabel('收盘价（元）')
ax.grid(alpha=0.3)

# 格式化日期轴
ax.xaxis.set_major_formatter(mdates.DateFormatter('%Y-%m'))
plt.setp(ax.xaxis.get_majorticklabels(), rotation=45)

plt.title('黄金ETF价格走势与金字塔操作点标记')
plt.tight_layout()
plt.show()

# ---------------------- 策略结果统计 ----------------------
final_asset = fund_etf_hist_em_df.iloc[-1]['总资产']
total_return = (final_asset - initial_cash) / initial_cash
max_drawdown = (fund_etf_hist_em_df['总资产'].cummax() - fund_etf_hist_em_df['总资产']).max() / initial_cash

print(f"策略初始资金：{initial_cash}元")
print(f"策略最终总资产：{final_asset:.2f}元")
print(f"总收益率：{total_return:.2%}")
print(f"最大回撤：{max_drawdown:.2%}")

# 输出关键操作记录（最后20条）
print("\n最后20天操作记录：")
print(fund_etf_hist_em_df[['日期', '收盘', '持仓数量', '总资产', '操作']].tail(20))